#!/usr/bin/env python3
import imaplib, email, os, argparse
from utils.helpers import write_mail_bundle, build_mail_path
from utils.logger import setup_text_logger, JsonLogger

def list_folders(imap):
    """Liste alle IMAP-Ordner auf und gebe die Namen zurück."""
    typ, folders = imap.list()
    if typ != "OK":
        raise RuntimeError("IMAP LIST fehlgeschlagen")
    names = []
    for f in folders:
        line = f.decode()
        parts = line.split(' "/" ')
        if len(parts) >= 2:
            raw_name = parts[-1].strip('"')
        else:
            raw_name = line.split()[-1].strip('"')
        names.append(raw_name)
    return names

def fetch_folder(imap, folder, workdir, textlog, jsonlog, id_scheme, max_subject_len):
    """Hole alle Mails aus einem Ordner und speichere sie lokal."""
    textlog.info(f"Bearbeite Ordner: {folder}")
    sel = imap.select(folder, readonly=True)
    if sel[0] != "OK":
        textlog.error(f"Ordner kann nicht selektiert werden: {folder}")
        jsonlog.write("folder_error", {"folder": folder, "reason": "select_failed"})
        return

    typ, data = imap.search(None, "ALL")
    if typ != "OK":
        textlog.error(f"Suche in Ordner fehlgeschlagen: {folder}")
        jsonlog.write("folder_error", {"folder": folder, "reason": "search_failed"})
        return

    uids = data[0].split()
    jsonlog.write("folder_info", {"folder": folder, "count": len(uids)})

    for uid_b in uids:
        uid = uid_b.decode()
        try:
            typ, raw = imap.fetch(uid_b, "(RFC822)")
            if typ != "OK":
                raise RuntimeError("Fetch RFC822 fehlgeschlagen")

            msg = email.message_from_bytes(raw[0][1])
            mail_path = build_mail_path(msg, uid, id_scheme, max_subject_len)

            # Punkte im Ordnernamen als Unterordner interpretieren (z. B. Archiv.Internet.Hosting)
            folder_path = os.path.join(workdir, *folder.split('.'))
            maildir = os.path.join(folder_path, mail_path)
            os.makedirs(maildir, exist_ok=True)

            result = write_mail_bundle(msg, maildir)

            textlog.info(
                f"[OK] UID={uid} | Ordner={folder} | Pfad='{maildir}' | Betreff='{result['subject']}' | Anhänge={result['attachments_count']}"
            )
            jsonlog.write("mail_ok", {
                "folder": folder,
                "uid": uid,
                "mail_path": mail_path,
                "subject": result["subject"],
                "attachments_count": result["attachments_count"],
                "attachments": result["attachments"]
            })
        except Exception as e:
            textlog.error(f"[FEHLER] UID={uid} | Ordner={folder} | {e}")
            jsonlog.write("mail_error", {
                "folder": folder,
                "uid": uid,
                "error": str(e)
            })

def main(args):
    os.makedirs(os.path.dirname(args.text_log), exist_ok=True)
    os.makedirs(os.path.dirname(args.json_log), exist_ok=True)

    textlog = setup_text_logger(args.text_log)
    jsonlog = JsonLogger(args.json_log)

    textlog.info("Starte Mail-Fetch")

    imap = imaplib.IMAP4_SSL(args.server)
    imap.login(args.user, args.password)

    folders = list_folders(imap)
    for folder in folders:
        fetch_folder(imap, folder, args.workdir, textlog, jsonlog, args.id_scheme, args.max_subject_len)

    imap.logout()
    textlog.info("Mail-Fetch abgeschlossen")
    jsonlog.write("fetch_done", {})

if __name__ == "__main__":
    p = argparse.ArgumentParser()
    p.add_argument("--server", required=True)
    p.add_argument("--user", required=True)
    p.add_argument("--password", required=True)
    p.add_argument("--workdir", required=True)
    p.add_argument("--text-log", required=True)
    p.add_argument("--json-log", required=True)
    p.add_argument("--id-scheme", default="DATE_SUBJECT")
    p.add_argument("--max-subject-len", type=int, default=120)
    args = p.parse_args()
    main(args)

