import os, html, re, email

def safe_filename(name):
    """Dateisystem-sichere Umwandlung eines Namens."""
    if not name:
        return "unnamed"
    cleaned = re.sub(r'[^a-zA-Z0-9._-]+', '_', name).strip('_')
    return cleaned or "unnamed"

def truncate(s, maxlen):
    """Begrenze Länge, entferne trailing underscores."""
    if maxlen and len(s) > maxlen:
        return s[:maxlen].rstrip('_')
    return s

def build_mail_path(msg, uid, scheme="DATE_SUBJECT", max_subject_len=120):
    """Erzeuge Jahr/Monat/ID-Pfad mit Datum-Bindestrichen und bereinigtem Betreff."""
    date_hdr = msg.get("Date")
    dt = None
    try:
        if date_hdr:
            dt = email.utils.parsedate_to_datetime(date_hdr)
    except Exception:
        dt = None

    year = dt.strftime("%Y") if dt else "0000"
    month = dt.strftime("%m") if dt else "00"
    day = dt.strftime("%d") if dt else "00"

    subject = msg.get("Subject", "")
    safe_subject = truncate(safe_filename(subject), max_subject_len)

    if scheme == "SUBJECT":
        mail_id = safe_subject or f"Mail_{uid}"
    elif scheme == "DATE_SUBJECT":
        stamp = f"{year}-{month}-{day}"
        base = safe_subject or f"Mail_{uid}"
        mail_id = f"{stamp}_{base}"
    else:
        mail_id = uid

    return os.path.join(year, month, mail_id)

def extract_parts(msg):
    """Extrahiere HTML/Text-Body und Anhänge aus einer MIME-Mail."""
    html_body, text_body, attachments = None, None, []
    if msg.is_multipart():
        for part in msg.walk():
            ctype = part.get_content_type()
            disp = str(part.get_content_disposition() or "")
            filename = part.get_filename()
            if ctype == "text/html" and html_body is None:
                payload = part.get_payload(decode=True) or b""
                html_body = payload.decode(part.get_content_charset() or "utf-8", errors="ignore")
            elif ctype == "text/plain" and text_body is None:
                payload = part.get_payload(decode=True) or b""
                text_body = payload.decode(part.get_content_charset() or "utf-8", errors="ignore")
            elif filename or disp == "attachment":
                data = part.get_payload(decode=True) or b""
                attachments.append((filename or "attachment", data))
    else:
        ctype = msg.get_content_type()
        payload = msg.get_payload(decode=True) or b""
        if ctype == "text/html":
            html_body = payload.decode(msg.get_content_charset() or "utf-8", errors="ignore")
        elif ctype == "text/plain":
            text_body = payload.decode(msg.get_content_charset() or "utf-8", errors="ignore")
    return html_body, text_body, attachments

def write_mail_bundle(msg, maildir):
    """Schreibe HTML-Ansicht, raw.eml, Anhänge und meta.json in den Zielordner."""
    os.makedirs(maildir, exist_ok=True)

    subject = msg.get("Subject", "(kein Betreff)")
    from_ = msg.get("From", "")
    to_ = msg.get("To", "")
    cc_ = msg.get("Cc", "")
    date_ = msg.get("Date", "")
    mid_ = msg.get("Message-ID", "")

    html_body, text_body, attachments = extract_parts(msg)
    if html_body is None:
        html_body = f"<pre>{html.escape(text_body or '')}</pre>"

    header_html = (
        f"<div style='font-family:system-ui,Segoe UI,Arial; font-size:14px; "
        f"border-bottom:1px solid #ddd; padding-bottom:8px; margin-bottom:12px;'>"
        f"<div><strong>Betreff:</strong> {html.escape(subject or '')}</div>"
        f"<div><strong>Von:</strong> {html.escape(from_ or '')}</div>"
        f"<div><strong>An:</strong> {html.escape(to_ or '')}</div>"
        f"<div><strong>Cc:</strong> {html.escape(cc_ or '')}</div>"
        f"<div><strong>Datum:</strong> {html.escape(date_ or '')}</div>"
        f"<div><strong>Message-ID:</strong> {html.escape(mid_ or '')}</div>"
        f"</div>"
    )

    with open(os.path.join(maildir, "mail.html"), "w", encoding="utf-8") as f:
        f.write("<!doctype html><meta charset='utf-8'>")
        f.write(header_html)
        f.write(html_body)

    with open(os.path.join(maildir, "raw.eml"), "wb") as f:
        f.write(msg.as_bytes())

    saved_attachments = []
    for (fname, data) in attachments:
        fn = safe_filename(fname)
        path = os.path.join(maildir, fn)
        with open(path, "wb") as af:
            af.write(data)
        saved_attachments.append(fn)

    meta = {
        "subject": subject,
        "from": from_,
        "to": to_,
        "cc": cc_,
        "date": date_,
        "message_id": mid_,
        "attachments": saved_attachments,
    }
    with open(os.path.join(maildir, "meta.json"), "w", encoding="utf-8") as mf:
        import json
        mf.write(json.dumps(meta, ensure_ascii=False, indent=2))

    return {
        "subject": subject,
        "attachments_count": len(saved_attachments),
        "attachments": saved_attachments
    }

